import asyncio
import json

from pylog.pylogger import PyLogger
from py_pli.pylib import send_msg
from py_pli.pylib import VUnits

import time as t
from datetime import datetime as DT

from virtualunits.HAL import HAL

from py_pli.pylib import URPCInterfaceType
from py_pli.pylib import URPCFunctions

from virtualunits.vu_node_application import VUNodeApplication
from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from fleming.common.node_io import FMBAnalogOutput
from fleming.common.node_io import EEFAnalogInput
from fleming.common.node_io import EEFAnalogOutput
from fleming.common.node_io import EEFDigitalOutput

from urpc.nodefunctions import NodeFunctions

from fleming.common.firmware_util import *

hal_unit: HAL = VUnits.instance.hal

fmb_unit: VUNodeApplication = hal_unit.nodes['Mainboard']

fmb_endpoint: NodeFunctions = fmb_unit.endpoint


# Node Endpoint ################0########################################################################################

nodes = {
    'eef'       : {'id':0x0008, 'delay':5},
    'mc6'       : {'id':0x0010, 'delay':1},
    'mc6_stk'   : {'id':0x0028, 'delay':1},
    'fmb'       : {'id':0x0018, 'delay':1},
    'pmc'       : {'id':0x00F0, 'delay':1},
    'pmc1'      : {'id':0x00F1, 'delay':1},
    'pmc2'      : {'id':0x00F2, 'delay':1},
    'pmc3'      : {'id':0x00F3, 'delay':1},
}

pd_channel = {
    'ref' : MeasurementChannel.REF,
    'abs' : MeasurementChannel.ABS,
    'aux' : MeasurementChannel.AUX,
}

pd_sample = {
    'ref' : TriggerSignal.SampleRef,
    'abs' : TriggerSignal.SampleAbs,
    'aux' : TriggerSignal.SampleAux,
}

pd_reset = {
    'ref' : OutputSignal.Deprc_IntRstRef,
    'abs' : OutputSignal.Deprc_IntRstAbs,
    'aux' : OutputSignal.Deprc_IntRstAux,
}

integrator_mode = {
    'reset'     : IntegratorMode.full_reset,
    'low_reset' : IntegratorMode.low_range_reset,
    'auto'      : IntegratorMode.integrate_autorange,
    'fixed'     : IntegratorMode.integrate_with_fixed_range,
    'low'       : IntegratorMode.integrate_in_low_range,
    'high'      : IntegratorMode.integrate_in_high_range,
}

ANALOGCONTROLMODE_ADD_TO_HIGH_OFFSET            = 1
ANALOGCONTROLMODE_ADD_TO_LOW_OFFSET             = 2
ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET    = 3

# PDD Measurement Control #########################################################################################

async def pd_measure_v7(window_us, name = 'ref', mode = 'auto', fixed_range_us=20, offset=0, wait_for_trigger=0):
    meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit
    full_reset_delay = 10000000     # 100 ms (min 400 us)
    conversion_delay = 1200         #  12 us
    trigger_delay = 100000 - 1000   # 990 us (1 kHz signal, starting 10 us after window)
    switch_delay = 25               # 250 ns delay to counter the charge injection

    if (name not in pd_channel):
        raise ValueError(f"name must be {pd_channel}")
    if (mode not in integrator_mode):
        raise ValueError(f"mode must be {integrator_mode}")
    if (window_us < 12) or (window_us > 671088):
        raise ValueError(f"window_us must be in the range [12, 671088] us")
    if (fixed_range_us < 0) or (fixed_range_us > 671088):
        raise ValueError(f"fixed_range_us must be in the range [0, 671088] us")
        
    window      = round(window_us * 100)
    fixed_range = round(fixed_range_us * 100)
    if (fixed_range >= window):
        fixed_range = 0

    op_id = 'pd_measure'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # high_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=2)   # low_offset
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=3)   # high_offset
    # Reset the offsets
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.full_offset_reset})
    # Set D(0) to the negation of the offset (16 bit two's complement)
    seq_gen.LoadDataReg(stackNotRegDst=False, dstReg=0, value=(-offset & 0xFFFF))
    # Enable full reset
    seq_gen.TimerWaitAndRestart(full_reset_delay - conversion_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    # Measure the high range offset
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.read_offset})
    # Wait for tigger
    if (wait_for_trigger == 1):
        seq_gen.TimerWait()
        seq_gen.WaitForTriggerInput(TRF=True)
        seq_gen.TimerWaitAndRestart(trigger_delay - switch_delay)
    # Start the integrator in low range (250 ns before the measurement window starts -> TBD)
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.low_range_reset})
    # Measure the low range offset, this is the start of the measurement window
    seq_gen.TimerWaitAndRestart(window - fixed_range)
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.SetIntegratorMode(**{name: integrator_mode[mode]})
    if (fixed_range > 0):
        # Switch to fixed range integration
        seq_gen.TimerWaitAndRestart(fixed_range)
        seq_gen.SetIntegratorMode(**{name: IntegratorMode.integrate_with_fixed_range})
    # Load the measured offset into the register
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=2)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=3)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.read_offset})
    # Add negation of offset to both offset registers
    seq_gen.SetAnalogControl(**{name: ANALOGCONTROLMODE_ADD_TO_LOW_AND_HIGH_OFFSET})
    # Measure the result
    seq_gen.TimerWait()
    seq_gen.SetTriggerOutput(pd_sample[name])
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=False, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(pd_channel[name], isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=False, dword=False, addrPos=0, resultPos=1)
    # Enable full reset
    seq_gen.SetIntegratorMode(**{name: IntegratorMode.full_reset})
    # Reset the offsets (only needed in case a sequence without offset correction is called after this)
    seq_gen.SetAnalogControl(**{name: AnalogControlMode.full_offset_reset})
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 4)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    results = await meas_unit.ReadMeasurementValues(op_id)

    #PyLogger.logger.info(f"Low: {results[0]:5d} ; High: {results[1]:5d} ; (Offset: {results[2]:5d} ; {results[3]:5d})")

    return results

async def fmb_set_led_current(current=17, source='fmb', channel='led1', led_type='green'):
    bc = fmb_led_channel[channel]['bc']
    gs = fmb_led_channel[channel]['gs']
    full_scale_current = fmb_full_scale_current[source]
    brightness = current / full_scale_current
    current = round(min(int(brightness * 128), 127) / 127 * full_scale_current)
    if (current > 0):
        await fmb_endpoint.SetAnalogOutput(bc, brightness)
        await fmb_endpoint.SetAnalogOutput(gs, 1.0)
    else:
        await fmb_endpoint.SetAnalogOutput(bc, 0.0)
        await fmb_endpoint.SetAnalogOutput(gs, 0.0)

    await asyncio.sleep(0.2)
    return current

async def set_led_current(current, source='fmb', channel='led5', led_type='red'):
    if source.startswith('fmb'):
        return await fmb_set_led_current(current, source, channel, led_type)
    
    raise Exception(f"Invalid source: {source}")

fmb_led_channel = {
    'led1' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG1},
    'led2' : {'bc':FMBAnalogOutput.BCR, 'gs':FMBAnalogOutput.GSR3},
    'led3' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG2},
    'led4' : {'bc':FMBAnalogOutput.BCB, 'gs':FMBAnalogOutput.GSB3},
    'led5' : {'bc':FMBAnalogOutput.BCG, 'gs':FMBAnalogOutput.GSG3},
}

fmb_full_scale_current = {
    'fmb'    :   127,   # Brightness Value [0, 127]
    'fmb#7'  :  2129,   # Measured with Keithley Multimeter (FMB#7,  24k3, Rev 2)
    'fmb#12' :  2084,   # Measured with Keithley Multimeter (FMB#12, 24k3, Rev 2)
    'fmb#15' : 11084,   # Measured with Keithley Multimeter (FMB#15,  4k7, Rev 2)
}

async def pd_pcb_test():

    await send_msg(json.dumps({'result': f"Photodiode Dynamic Detection Test starting..."}))

    await start_firmware('fmb')
    fmb = get_node_endpoint('fmb')
    await fmb.SetDigitalOutput(0, 0, timeout=1)
    await asyncio.sleep(0.3)
    await start_firmware('eef')
        
    checksum = 0 
    
    lr_single = 0
    lr = 0
    hr_single = 0
    hr = 0
    
    #Dunkelmessung
    await set_led_current(0)    
    
    for i in range(10):
        results = await pd_measure_v7(500, name='ref', mode='auto', fixed_range_us=20, offset=0, wait_for_trigger=0)
        lr_single = lr_single + results[0]
        lr = lr_single / 10     
    if lr < 5 and lr > 0:
        PyLogger.logger.info(f"CHECK      Low Value:    {lr}    High Value:    {hr}")
        await send_msg(json.dumps({'result': f"Dark Measurement DONE..."}))
        checksum = checksum + 1
    else:
        PyLogger.logger.info(f"Low Value:    {lr}    High Value:    {hr}       FAILED: Dark Value too High or Zero") 
        await send_msg(json.dumps({'result': f"Dark Measurement FAILED..."}))
            
    lr_single = 0
    lr = 0
    hr_single = 0
    hr = 0
    
    #Low Range Messung
    await set_led_current(1)
    
    for i in range(10):
        results = await pd_measure_v7(2150, name='ref', mode='auto', fixed_range_us=20, offset=0, wait_for_trigger=0)
        lr_single = lr_single + results[0]
        lr = lr_single / 10  
        hr_single = hr_single + results[1]
        hr = hr_single / 10  
    if lr > 27500 and lr < 30500 and hr == 0:
        PyLogger.logger.info(f"CHECK      Low Value:    {lr}    High Value:    {hr}")
        await send_msg(json.dumps({'result': f"Low Range Measurement DONE..."}))
        checksum = checksum + 10
    else:
        PyLogger.logger.info(f"Low Value:    {lr}    High Value:    {hr}        FAILED:")
        await send_msg(json.dumps({'result': f"Low Range Measurement FAILED..."}))
        
    lr_single = 0
    lr = 0
    hr_single = 0
    hr = 0
    
    #High Range Messung
    await set_led_current(100)
    
    for i in range(10):
        results = await pd_measure_v7(1200, name='ref', mode='auto', fixed_range_us=20, offset=0, wait_for_trigger=0)
        lr_single = lr_single + results[0]
        lr = lr_single / 10  
        hr_single = hr_single + results[1]
        hr = hr_single / 10  
    if hr > 14500 and hr < 17500 and lr == 0:
        PyLogger.logger.info(f"CHECK      Low Value:    {lr}    High Value:    {hr}")
        await send_msg(json.dumps({'result': f"High Range Measurement DONE..."}))
        checksum = checksum + 100
    else:
        PyLogger.logger.info(f"Low Value:    {lr}    High Value:    {hr}       FAILED")
        await send_msg(json.dumps({'result': f"High Range Measurement FAILED..."}))
     
    lr_single = 0
    lr = 0
    hr_single = 0
    hr = 0
    
    #ADC Full Scale Messung
    await set_led_current(100)
    
    for i in range(10):
        results = await pd_measure_v7(5500, name='ref', mode='auto', fixed_range_us=20, offset=0, wait_for_trigger=0)
        lr_single = lr_single + results[0]
        lr = lr_single / 10  
        hr_single = hr_single + results[1]
        hr = hr_single / 10  
    if hr > 62500 and hr < 65500 and lr == 0:
        PyLogger.logger.info(f"CHECK      Low Value:    {lr}    High Value:    {hr}")
        await send_msg(json.dumps({'result': f"Full Scale Measurement DONE..."}))
        checksum = checksum + 1000
    else:
        PyLogger.logger.info(f"Low Value:    {lr}    High Value:    {hr}       FAILED")
        await send_msg(json.dumps({'result': f"Full Scale Measurement FAILED..."}))
            
    # Turn LED Off
    await set_led_current(0)
    await fmb.SetDigitalOutput(0, 1, timeout=1)
    
    if checksum == 1111:
        PyLogger.logger.info(f"Photodiode Dynamic Detection Test SUCCESSFUL! DUT save to remove.")
        await send_msg(json.dumps({'result': f"Photodiode Dynamic Detection Test SUCCESSFUL! DUT save to remove."}))
    else:
        PyLogger.logger.info(f"Photodiode Dynamic Detection Test FAILED with ERRORCODE {checksum}! Check Test Conditions.")
        await send_msg(json.dumps({'result': f"Photodiode Dynamic Detection Test FAILED!"}))
        await send_msg(json.dumps({'result': f"ERRORCODE: {checksum}!"}))
        await send_msg(json.dumps({'result': f"See Documentation for more Information & Check Test Conditions."}))

# PDD Reference Signal Control #########################################################################################

